<?php

namespace Tests\Feature\Api\V1;

use App\Models\User;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class WalletTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected string $token;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => Hash::make('password123'),
        ]);
        
        Wallet::create(['user_id' => $this->user->id, 'balance' => 50]);
        $this->token = $this->user->createToken('test-token')->plainTextToken;
    }

    /**
     * Test user can get wallet balance
     */
    public function test_user_can_get_balance(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/wallet/balance');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => ['balance', 'auto_topup_enabled'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can top up wallet
     */
    public function test_user_can_top_up_wallet(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/wallet/topup', [
                'amount' => 50,
                'payment_method' => 'card',
            ]);

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => ['balance'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can get transaction history
     */
    public function test_user_can_get_transactions(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/wallet/transactions');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data',
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can update auto top-up settings
     */
    public function test_user_can_update_auto_topup(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson('/api/v1/wallet/auto-topup', [
                'auto_topup_enabled' => true,
                'auto_topup_threshold' => 10,
                'auto_topup_amount' => 20,
            ]);

        $response->assertStatus(200)
            ->assertJson(['success' => true]);
    }
}
