<?php

namespace Tests\Feature\Api\V1;

use App\Models\User;
use App\Models\Vehicle;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class VehicleTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected string $token;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => Hash::make('password123'),
        ]);
        
        $this->token = $this->user->createToken('test-token')->plainTextToken;
    }

    /**
     * Test user can create a vehicle
     */
    public function test_user_can_create_vehicle(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/vehicles', [
                'type' => 'car',
                'make' => 'Toyota',
                'model' => 'Corolla',
                'registration_number' => 'AB12CDE',
                'year' => 2020,
                'color' => 'Red',
            ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'data' => ['id', 'type', 'make', 'model'],
                'message',
            ])
            ->assertJson(['success' => true]);

        $this->assertDatabaseHas('vehicles', [
            'user_id' => $this->user->id,
            'registration_number' => 'AB12CDE',
        ]);
    }

    /**
     * Test user can list vehicles
     */
    public function test_user_can_list_vehicles(): void
    {
        Vehicle::factory()->create(['user_id' => $this->user->id]);
        Vehicle::factory()->create(['user_id' => $this->user->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/vehicles');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [['id', 'type', 'make', 'model']],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can view a vehicle
     */
    public function test_user_can_view_vehicle(): void
    {
        $vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson("/api/v1/vehicles/{$vehicle->id}");

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => ['id', 'type', 'make', 'model'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can update a vehicle
     */
    public function test_user_can_update_vehicle(): void
    {
        $vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson("/api/v1/vehicles/{$vehicle->id}", [
                'color' => 'Blue',
            ]);

        $response->assertStatus(200)
            ->assertJson(['success' => true]);

        $this->assertDatabaseHas('vehicles', [
            'id' => $vehicle->id,
            'color' => 'Blue',
        ]);
    }

    /**
     * Test user can delete a vehicle
     */
    public function test_user_can_delete_vehicle(): void
    {
        $vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->deleteJson("/api/v1/vehicles/{$vehicle->id}");

        $response->assertStatus(200)
            ->assertJson(['success' => true]);

        $this->assertDatabaseMissing('vehicles', ['id' => $vehicle->id]);
    }
}
