<?php

namespace Tests\Feature\Api\V1;

use App\Models\Policy;
use App\Models\Session;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class SessionTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected string $token;
    protected Policy $policy;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => Hash::make('password123'),
            'kyc_status' => 'verified',
        ]);
        
        Wallet::create(['user_id' => $this->user->id, 'balance' => 100]);
        $vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);
        $this->policy = Policy::factory()->create([
            'user_id' => $this->user->id,
            'vehicle_id' => $vehicle->id,
            'product_type' => 'PAYG',
            'status' => 'active',
        ]);
        
        $this->token = $this->user->createToken('test-token')->plainTextToken;
    }

    /**
     * Test user can start session
     */
    public function test_user_can_start_session(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/sessions/start', [
                'policy_id' => $this->policy->id,
            ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'data' => ['id', 'status', 'started_at'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user cannot start session with insufficient balance
     */
    public function test_user_cannot_start_session_with_low_balance(): void
    {
        $wallet = $this->user->wallet;
        $wallet->update(['balance' => 5]); // Below minimum £10

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/sessions/start', [
                'policy_id' => $this->policy->id,
            ]);

        $response->assertStatus(400)
            ->assertJson(['success' => false]);
    }

    /**
     * Test user can stop session
     */
    public function test_user_can_stop_session(): void
    {
        $session = Session::factory()->create([
            'policy_id' => $this->policy->id,
            'user_id' => $this->user->id,
            'status' => 'active',
            'started_at' => now()->subHour(),
        ]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/sessions/{$session->id}/stop");

        $response->assertStatus(200)
            ->assertJson(['success' => true]);

        $this->assertDatabaseHas('payg_sessions', [
            'id' => $session->id,
            'status' => 'completed',
        ]);
    }

    /**
     * Test user can get active session
     */
    public function test_user_can_get_active_session(): void
    {
        $session = Session::factory()->create([
            'policy_id' => $this->policy->id,
            'user_id' => $this->user->id,
            'status' => 'active',
            'started_at' => now()->subHour(),
        ]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/sessions/active');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => ['id', 'status'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }
}
