<?php

namespace Tests\Feature\Api\V1;

use App\Models\Policy;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class PolicyTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected string $token;
    protected Vehicle $vehicle;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => Hash::make('password123'),
            'kyc_status' => 'verified',
        ]);
        
        Wallet::create(['user_id' => $this->user->id, 'balance' => 100]);
        $this->vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);
        $this->token = $this->user->createToken('test-token')->plainTextToken;
    }

    /**
     * Test user can get quote
     */
    public function test_user_can_get_quote(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/policies/quote', [
                'product_type' => 'PAYG',
                'vehicle_id' => $this->vehicle->id,
            ]);

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => ['product_type', 'premium_amount'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can create policy
     */
    public function test_user_can_create_policy(): void
    {
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/policies', [
                'product_type' => 'PAYG',
                'vehicle_id' => $this->vehicle->id,
            ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'data' => ['id', 'product_type', 'status'],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user cannot create policy without KYC verification
     */
    public function test_user_cannot_create_policy_without_kyc(): void
    {
        $user = User::factory()->create(['kyc_status' => 'pending']);
        $token = $user->createToken('test-token')->plainTextToken;
        $vehicle = Vehicle::factory()->create(['user_id' => $user->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->postJson('/api/v1/policies', [
                'product_type' => 'PAYG',
                'vehicle_id' => $vehicle->id,
            ]);

        $response->assertStatus(403)
            ->assertJson(['success' => false]);
    }

    /**
     * Test user can list policies
     */
    public function test_user_can_list_policies(): void
    {
        Policy::factory()->create(['user_id' => $this->user->id, 'vehicle_id' => $this->vehicle->id]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/policies');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data' => [['id', 'product_type', 'status']],
                'message',
            ])
            ->assertJson(['success' => true]);
    }

    /**
     * Test user can activate policy
     */
    public function test_user_can_activate_policy(): void
    {
        $policy = Policy::factory()->create([
            'user_id' => $this->user->id,
            'vehicle_id' => $this->vehicle->id,
            'status' => 'draft',
        ]);

        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/policies/{$policy->id}/activate");

        $response->assertStatus(200)
            ->assertJson(['success' => true]);

        $this->assertDatabaseHas('policies', [
            'id' => $policy->id,
            'status' => 'active',
        ]);
    }
}
