<?php

namespace Tests\Feature\Api\V1;

use App\Models\Policy;
use App\Models\Session;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

/**
 * Complete API Test Suite
 * Tests all endpoints from the plan
 */
class CompleteApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected string $token;
    protected Vehicle $vehicle;
    protected Policy $policy;
    protected Wallet $wallet;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create verified user with all required data
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => Hash::make('password123'),
            'kyc_status' => 'verified',
        ]);
        
        $this->wallet = Wallet::create([
            'user_id' => $this->user->id,
            'balance' => 100,
        ]);
        
        $this->vehicle = Vehicle::factory()->create(['user_id' => $this->user->id]);
        
        $this->policy = Policy::factory()->create([
            'user_id' => $this->user->id,
            'vehicle_id' => $this->vehicle->id,
            'product_type' => 'PAYG',
            'status' => 'active',
        ]);
        
        $this->token = $this->user->createToken('test-token')->plainTextToken;
    }

    /**
     * Test Authentication Endpoints
     */
    public function test_authentication_endpoints(): void
    {
        // Register
        $registerResponse = $this->postJson('/api/v1/auth/register', [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password123',
        ]);
        $registerResponse->assertStatus(201)->assertJson(['success' => true]);

        // Login
        $loginResponse = $this->postJson('/api/v1/auth/login', [
            'email' => 'test@example.com',
            'password' => 'password123',
        ]);
        $loginResponse->assertStatus(200)->assertJson(['success' => true]);

        // Logout
        $logoutResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/auth/logout');
        $logoutResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test User Management Endpoints
     */
    public function test_user_management_endpoints(): void
    {
        // Get profile
        $profileResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/user/profile');
        $profileResponse->assertStatus(200)->assertJson(['success' => true]);

        // Update profile
        $updateResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson('/api/v1/user/profile', [
                'name' => 'Updated Name',
                'phone' => '+441234567890',
            ]);
        $updateResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test KYC Endpoints
     */
    public function test_kyc_endpoints(): void
    {
        // Get KYC status
        $statusResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/kyc/status');
        $statusResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Vehicle Endpoints
     */
    public function test_vehicle_endpoints(): void
    {
        // List vehicles
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/vehicles');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);

        // Create vehicle
        $createResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/vehicles', [
                'type' => 'car',
                'make' => 'Toyota',
                'model' => 'Corolla',
                'registration_number' => 'TEST123',
                'year' => 2020,
            ]);
        $createResponse->assertStatus(201)->assertJson(['success' => true]);

        $vehicleId = $createResponse->json('data.id');

        // Get vehicle
        $getResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson("/api/v1/vehicles/{$vehicleId}");
        $getResponse->assertStatus(200)->assertJson(['success' => true]);

        // Update vehicle
        $updateResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson("/api/v1/vehicles/{$vehicleId}", [
                'color' => 'Blue',
            ]);
        $updateResponse->assertStatus(200)->assertJson(['success' => true]);

        // Delete vehicle
        $deleteResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->deleteJson("/api/v1/vehicles/{$vehicleId}");
        $deleteResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Policy Endpoints
     */
    public function test_policy_endpoints(): void
    {
        // Get quote
        $quoteResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/policies/quote', [
                'product_type' => 'PAYG',
                'vehicle_id' => $this->vehicle->id,
            ]);
        $quoteResponse->assertStatus(200)->assertJson(['success' => true]);

        // Create policy
        $createResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/policies', [
                'product_type' => '30D',
                'vehicle_id' => $this->vehicle->id,
            ]);
        $createResponse->assertStatus(201)->assertJson(['success' => true]);

        $policyId = $createResponse->json('data.id');

        // List policies
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/policies');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);

        // Get policy
        $getResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson("/api/v1/policies/{$policyId}");
        $getResponse->assertStatus(200)->assertJson(['success' => true]);

        // Activate policy
        $activateResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/policies/{$policyId}/activate");
        $activateResponse->assertStatus(200)->assertJson(['success' => true]);

        // Deactivate policy
        $deactivateResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/policies/{$policyId}/deactivate");
        $deactivateResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Session Endpoints
     */
    public function test_session_endpoints(): void
    {
        // Start session
        $startResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/sessions/start', [
                'policy_id' => $this->policy->id,
            ]);
        $startResponse->assertStatus(201)->assertJson(['success' => true]);

        $sessionId = $startResponse->json('data.id');

        // Get active session
        $activeResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/sessions/active');
        $activeResponse->assertStatus(200)->assertJson(['success' => true]);

        // Update location
        $locationResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/sessions/{$sessionId}/location", [
                'latitude' => 51.5074,
                'longitude' => -0.1278,
            ]);
        $locationResponse->assertStatus(200)->assertJson(['success' => true]);

        // Stop session
        $stopResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/sessions/{$sessionId}/stop");
        $stopResponse->assertStatus(200)->assertJson(['success' => true]);

        // List sessions
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/sessions');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Wallet Endpoints
     */
    public function test_wallet_endpoints(): void
    {
        // Get balance
        $balanceResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/wallet/balance');
        $balanceResponse->assertStatus(200)->assertJson(['success' => true]);

        // Top up
        $topupResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/wallet/topup', [
                'amount' => 50,
                'payment_method' => 'card',
            ]);
        $topupResponse->assertStatus(200)->assertJson(['success' => true]);

        // Get transactions
        $transactionsResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/wallet/transactions');
        $transactionsResponse->assertStatus(200)->assertJson(['success' => true]);

        // Update auto top-up
        $autoTopupResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson('/api/v1/wallet/auto-topup', [
                'auto_topup_enabled' => true,
                'auto_topup_threshold' => 10,
                'auto_topup_amount' => 20,
            ]);
        $autoTopupResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Document Endpoints
     */
    public function test_document_endpoints(): void
    {
        // List documents
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/documents');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Claim Endpoints
     */
    public function test_claim_endpoints(): void
    {
        // Create claim
        $createResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/claims', [
                'policy_id' => $this->policy->id,
                'description' => 'Test accident claim',
                'incident_date' => now()->format('Y-m-d'),
                'incident_location' => 'London, UK',
            ]);
        $createResponse->assertStatus(201)->assertJson(['success' => true]);

        $claimId = $createResponse->json('data.id');

        // List claims
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/claims');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);

        // Get claim
        $getResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson("/api/v1/claims/{$claimId}");
        $getResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Work Provider Endpoints
     */
    public function test_work_provider_endpoints(): void
    {
        // Create work provider
        $createResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/work-providers', [
                'provider' => 'uber_eats',
                'provider_user_id' => 'UBER123',
                'provider_token' => 'token123',
            ]);
        $createResponse->assertStatus(201)->assertJson(['success' => true]);

        $providerId = $createResponse->json('data.id');

        // List providers
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/work-providers');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);

        // Link provider
        $linkResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/work-providers/{$providerId}/link");
        $linkResponse->assertStatus(200)->assertJson(['success' => true]);

        // Unlink provider
        $unlinkResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson("/api/v1/work-providers/{$providerId}/unlink");
        $unlinkResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Notification Endpoints
     */
    public function test_notification_endpoints(): void
    {
        // List notifications
        $listResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/notifications');
        $listResponse->assertStatus(200)->assertJson(['success' => true]);

        // Register token
        $tokenResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->postJson('/api/v1/notifications/token', [
                'fcm_token' => 'test_fcm_token_123',
            ]);
        $tokenResponse->assertStatus(200)->assertJson(['success' => true]);

        // Update preferences
        $prefsResponse = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->putJson('/api/v1/notifications/preferences', []);
        $prefsResponse->assertStatus(200)->assertJson(['success' => true]);
    }

    /**
     * Test Unified Response Format
     */
    public function test_unified_response_format(): void
    {
        // Test success response
        $response = $this->withHeader('Authorization', 'Bearer ' . $this->token)
            ->getJson('/api/v1/user/profile');
        
        $response->assertStatus(200)
            ->assertJsonStructure([
                'success',
                'data',
                'message',
            ])
            ->assertJson(['success' => true]);

        // Test validation error
        $validationResponse = $this->postJson('/api/v1/auth/register', [
            'name' => '',
            'email' => 'invalid',
        ]);
        $validationResponse->assertStatus(422)
            ->assertJsonStructure([
                'success',
                'message',
                'errors',
            ])
            ->assertJson(['success' => false]);
    }
}

