<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use OpenApi\Attributes as OA;

class WalletController extends Controller
{
    use ApiResponse;

    /**
     * Get wallet balance
     */
    #[OA\Get(
        path: '/api/v1/wallet/balance',
        summary: 'Get wallet balance',
        description: 'Retrieve the current wallet balance and auto top-up settings',
        tags: ['Wallet'],
        security: [['sanctum' => []]],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Balance retrieved successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'balance', type: 'number', example: 570.00),
                            new OA\Property(property: 'auto_topup_enabled', type: 'boolean', example: false),
                            new OA\Property(property: 'auto_topup_threshold', type: 'number', example: 10.00),
                            new OA\Property(property: 'auto_topup_amount', type: 'number', example: 20.00),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Balance retrieved successfully'),
                    ]
                )
            ),
        ]
    )]
    public function balance(Request $request): JsonResponse
    {
        $wallet = $request->user()->wallet;
        
        if (!$wallet) {
            $wallet = Wallet::create([
                'user_id' => $request->user()->id,
                'balance' => 0,
            ]);
        }

        return $this->successResponse([
            'balance' => $wallet->balance,
            'auto_topup_enabled' => $wallet->auto_topup_enabled,
            'auto_topup_threshold' => $wallet->auto_topup_threshold,
            'auto_topup_amount' => $wallet->auto_topup_amount,
        ], 'Balance retrieved successfully');
    }

    /**
     * Top up wallet
     */
    #[OA\Post(
        path: '/api/v1/wallet/topup',
        summary: 'Top up wallet',
        description: 'Add funds to the wallet. Payment processing via Stripe (placeholder)',
        tags: ['Wallet'],
        security: [['sanctum' => []]],
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\JsonContent(
                required: ['amount'],
                properties: [
                    new OA\Property(property: 'amount', type: 'number', minimum: 5, maximum: 1000, example: 50.00),
                    new OA\Property(property: 'payment_method', type: 'string', nullable: true, example: 'card'),
                ]
            )
        ),
        responses: [
            new OA\Response(
                response: 200,
                description: 'Wallet topped up successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'balance', type: 'number', example: 620.00),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Wallet topped up successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 422,
                description: 'Validation error',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Validation failed'),
                        new OA\Property(property: 'errors', type: 'object'),
                    ]
                )
            ),
        ]
    )]
    public function topUp(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:5|max:1000',
            'payment_method' => 'nullable|string|max:50',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $user = $request->user();
        $wallet = $user->wallet;
        
        if (!$wallet) {
            $wallet = Wallet::create(['user_id' => $user->id, 'balance' => 0]);
        }

        DB::transaction(function () use ($wallet, $request, $user) {
            $balanceBefore = $wallet->balance;
            $balanceAfter = $balanceBefore + $request->amount;

            $wallet->update(['balance' => $balanceAfter]);

            // TODO: Process actual payment via Stripe
            // For now, just create transaction record

            WalletTransaction::create([
                'wallet_id' => $wallet->id,
                'user_id' => $user->id,
                'type' => 'topup',
                'amount' => $request->amount,
                'balance_before' => $balanceBefore,
                'balance_after' => $balanceAfter,
                'payment_method' => $request->payment_method ?? 'card',
                'status' => 'completed',
                'reference' => 'TOPUP-' . strtoupper(uniqid()),
                'description' => 'Wallet top-up',
            ]);
        });

        $wallet->refresh();

        return $this->successResponse([
            'balance' => $wallet->balance,
        ], 'Wallet topped up successfully');
    }

    /**
     * Get transaction history
     */
    #[OA\Get(
        path: '/api/v1/wallet/transactions',
        summary: 'Get transaction history',
        description: 'Retrieve paginated list of wallet transactions',
        tags: ['Wallet'],
        security: [['sanctum' => []]],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Transactions retrieved successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'data', type: 'array', items: new OA\Items(type: 'object')),
                            new OA\Property(property: 'current_page', type: 'integer'),
                            new OA\Property(property: 'total', type: 'integer'),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Transactions retrieved successfully'),
                    ]
                )
            ),
        ]
    )]
    public function transactions(Request $request): JsonResponse
    {
        $transactions = WalletTransaction::where('user_id', $request->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return $this->successResponse($transactions, 'Transactions retrieved successfully');
    }

    /**
     * Update auto top-up settings
     */
    #[OA\Put(
        path: '/api/v1/wallet/auto-topup',
        summary: 'Update auto top-up settings',
        description: 'Configure automatic wallet top-up when balance falls below threshold',
        tags: ['Wallet'],
        security: [['sanctum' => []]],
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\JsonContent(
                required: ['auto_topup_enabled'],
                properties: [
                    new OA\Property(property: 'auto_topup_enabled', type: 'boolean', example: true),
                    new OA\Property(property: 'auto_topup_threshold', type: 'number', nullable: true, example: 10.00),
                    new OA\Property(property: 'auto_topup_amount', type: 'number', nullable: true, example: 20.00),
                ]
            )
        ),
        responses: [
            new OA\Response(
                response: 200,
                description: 'Auto top-up settings updated successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object'),
                        new OA\Property(property: 'message', type: 'string', example: 'Auto top-up settings updated successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 422,
                description: 'Validation error',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Validation failed'),
                        new OA\Property(property: 'errors', type: 'object'),
                    ]
                )
            ),
        ]
    )]
    public function updateAutoTopUp(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'auto_topup_enabled' => 'required|boolean',
            'auto_topup_threshold' => 'required_if:auto_topup_enabled,true|numeric|min:0',
            'auto_topup_amount' => 'required_if:auto_topup_enabled,true|numeric|min:5|max:1000',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $wallet = $request->user()->wallet;
        
        if (!$wallet) {
            $wallet = Wallet::create(['user_id' => $request->user()->id, 'balance' => 0]);
        }

        $wallet->update($request->only(['auto_topup_enabled', 'auto_topup_threshold', 'auto_topup_amount']));

        return $this->successResponse($wallet, 'Auto top-up settings updated successfully');
    }
}
