<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Document;
use App\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use OpenApi\Attributes as OA;

class DocumentController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of documents
     */
    #[OA\Get(
        path: '/api/v1/documents',
        summary: 'List all documents',
        description: 'Retrieve all active policy documents for the authenticated user',
        tags: ['Documents'],
        security: [['sanctum' => []]],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Documents retrieved successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'array', items: new OA\Items(type: 'object')),
                        new OA\Property(property: 'message', type: 'string', example: 'Documents retrieved successfully'),
                    ]
                )
            ),
        ]
    )]
    public function index(Request $request): JsonResponse
    {
        $documents = Document::where('user_id', $request->user()->id)
            ->where('is_active', true)
            ->with(['policy'])
            ->get();

        return $this->successResponse($documents, 'Documents retrieved successfully');
    }

    /**
     * Download document with signed URL
     */
    #[OA\Get(
        path: '/api/v1/documents/{id}/download',
        summary: 'Get document download URL',
        description: 'Generate a signed URL for downloading a policy document. URL is valid for 1 hour.',
        tags: ['Documents'],
        security: [['sanctum' => []]],
        parameters: [
            new OA\Parameter(name: 'id', in: 'path', required: true, schema: new OA\Schema(type: 'integer'), example: 1),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Download URL generated successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'download_url', type: 'string', example: 'https://example.com/storage/documents/...'),
                            new OA\Property(property: 'document', type: 'object'),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Download URL generated successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 404,
                description: 'Document not found',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Document not found'),
                    ]
                )
            ),
        ]
    )]
    public function download(Request $request, string $id): JsonResponse
    {
        $document = Document::where('id', $id)
            ->where('user_id', $request->user()->id)
            ->first();

        if (!$document) {
            return $this->notFoundResponse('Document not found');
        }

        if (!Storage::disk('public')->exists($document->file_path)) {
            return $this->notFoundResponse('Document file not found');
        }

        // Generate signed URL (valid for 1 hour)
        $url = Storage::disk('public')->temporaryUrl(
            $document->file_path,
            now()->addHour()
        );

        return $this->successResponse([
            'download_url' => $url,
            'document' => $document,
        ], 'Download URL generated successfully');
    }
}
