<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Claim;
use App\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use OpenApi\Attributes as OA;

class ClaimController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of claims
     */
    #[OA\Get(
        path: '/api/v1/claims',
        summary: 'List all claims',
        description: 'Retrieve paginated list of all insurance claims for the authenticated user',
        tags: ['Claims'],
        security: [['sanctum' => []]],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Claims retrieved successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'data', type: 'array', items: new OA\Items(type: 'object')),
                            new OA\Property(property: 'current_page', type: 'integer'),
                            new OA\Property(property: 'total', type: 'integer'),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Claims retrieved successfully'),
                    ]
                )
            ),
        ]
    )]
    public function index(Request $request): JsonResponse
    {
        $claims = Claim::where('user_id', $request->user()->id)
            ->with(['policy', 'session'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return $this->successResponse($claims, 'Claims retrieved successfully');
    }

    /**
     * Store a newly created claim
     */
    #[OA\Post(
        path: '/api/v1/claims',
        summary: 'Create a new claim',
        description: 'Report a new insurance claim with incident details',
        tags: ['Claims'],
        security: [['sanctum' => []]],
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\JsonContent(
                required: ['description', 'incident_date'],
                properties: [
                    new OA\Property(property: 'policy_id', type: 'integer', nullable: true, example: 1),
                    new OA\Property(property: 'session_id', type: 'integer', nullable: true, example: 1),
                    new OA\Property(property: 'description', type: 'string', example: 'Accident description'),
                    new OA\Property(property: 'incident_date', type: 'string', format: 'date', example: '2024-01-15'),
                    new OA\Property(property: 'incident_location', type: 'string', nullable: true, example: 'London, UK'),
                    new OA\Property(property: 'third_party_details', type: 'object', nullable: true),
                ]
            )
        ),
        responses: [
            new OA\Response(
                response: 201,
                description: 'Claim created successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object', properties: [
                            new OA\Property(property: 'claim_number', type: 'string', example: 'CLM-ABC123'),
                        ]),
                        new OA\Property(property: 'message', type: 'string', example: 'Claim created successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 422,
                description: 'Validation error',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Validation failed'),
                        new OA\Property(property: 'errors', type: 'object'),
                    ]
                )
            ),
        ]
    )]
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'policy_id' => 'nullable|exists:policies,id',
            'session_id' => 'nullable|exists:payg_sessions,id',
            'description' => 'required|string|max:5000',
            'incident_date' => 'required|date',
            'incident_location' => 'nullable|string|max:255',
            'third_party_details' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $claim = Claim::create([
            'user_id' => $request->user()->id,
            'policy_id' => $request->policy_id,
            'session_id' => $request->session_id,
            'claim_number' => 'CLM-' . strtoupper(uniqid()),
            'status' => 'pending',
            ...$request->only(['description', 'incident_date', 'incident_location', 'third_party_details']),
        ]);

        return $this->successResponse($claim, 'Claim created successfully', 201);
    }

    /**
     * Display the specified claim
     */
    #[OA\Get(
        path: '/api/v1/claims/{id}',
        summary: 'Get claim details',
        description: 'Retrieve details of a specific insurance claim',
        tags: ['Claims'],
        security: [['sanctum' => []]],
        parameters: [
            new OA\Parameter(name: 'id', in: 'path', required: true, schema: new OA\Schema(type: 'integer'), example: 1),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Claim retrieved successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object'),
                        new OA\Property(property: 'message', type: 'string', example: 'Claim retrieved successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 404,
                description: 'Claim not found',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Claim not found'),
                    ]
                )
            ),
        ]
    )]
    public function show(Request $request, string $id): JsonResponse
    {
        $claim = Claim::where('id', $id)
            ->where('user_id', $request->user()->id)
            ->with(['policy', 'session'])
            ->first();

        if (!$claim) {
            return $this->notFoundResponse('Claim not found');
        }

        return $this->successResponse($claim, 'Claim retrieved successfully');
    }

    /**
     * Upload evidence for claim
     */
    #[OA\Post(
        path: '/api/v1/claims/{id}/evidence',
        summary: 'Upload claim evidence',
        description: 'Upload evidence files (photos, documents) for a claim. Maximum 10 files.',
        tags: ['Claims'],
        security: [['sanctum' => []]],
        parameters: [
            new OA\Parameter(name: 'id', in: 'path', required: true, schema: new OA\Schema(type: 'integer'), example: 1),
        ],
        requestBody: new OA\RequestBody(
            required: true,
            content: new OA\MediaType(
                mediaType: 'multipart/form-data',
                schema: new OA\Schema(
                    required: ['files'],
                    properties: [
                        new OA\Property(property: 'files', type: 'array', items: new OA\Items(type: 'string', format: 'binary'), maxItems: 10),
                    ]
                )
            )
        ),
        responses: [
            new OA\Response(
                response: 200,
                description: 'Evidence uploaded successfully',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: true),
                        new OA\Property(property: 'data', type: 'object'),
                        new OA\Property(property: 'message', type: 'string', example: 'Evidence uploaded successfully'),
                    ]
                )
            ),
            new OA\Response(
                response: 404,
                description: 'Claim not found',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Claim not found'),
                    ]
                )
            ),
            new OA\Response(
                response: 422,
                description: 'Validation error',
                content: new OA\JsonContent(
                    properties: [
                        new OA\Property(property: 'success', type: 'boolean', example: false),
                        new OA\Property(property: 'message', type: 'string', example: 'Validation failed'),
                        new OA\Property(property: 'errors', type: 'object'),
                    ]
                )
            ),
        ]
    )]
    public function uploadEvidence(Request $request, string $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'files' => 'required|array|max:10',
            'files.*' => 'file|mimes:pdf,jpg,jpeg,png|max:10240',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $claim = Claim::where('id', $id)
            ->where('user_id', $request->user()->id)
            ->first();

        if (!$claim) {
            return $this->notFoundResponse('Claim not found');
        }

        $evidenceFiles = $claim->evidence_files ?? [];
        
        foreach ($request->file('files') as $file) {
            $path = $file->store('claim_evidence', 'public');
            $evidenceFiles[] = [
                'path' => $path,
                'name' => $file->getClientOriginalName(),
                'mime_type' => $file->getMimeType(),
                'size' => $file->getSize(),
                'uploaded_at' => now()->toIso8601String(),
            ];
        }

        $claim->update(['evidence_files' => $evidenceFiles]);

        return $this->successResponse($claim, 'Evidence uploaded successfully');
    }
}
